<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns:fmp="http://www.filemaker.com/fmpxmlresult" exclude-result-prefixes="fmp">
	<xsl:output method="html" version="1.0" encoding="UTF-8" indent="yes"/>
	<!--
File: subsummary.xsl

Transforms data exported in the FMPXMLRESULT grammar into an 
HTML table. Data is grouped according to values in the first
exported field. Members of each group are sorted by the second
field in the export.

The last field in the export is totaled if its data type is NUMBER,
otherwise the number of items in each group is counted.

In other words, data export in this manner:

Group   Last Name   First Name   Score
=======================================
Red     Smith       Jim          10
Red     Doe         John         20
Blue    Woods       Bill         12
Blue    Hope        Sue          17
Red     Ringer      Bob          18

Is presented this way:

Group   Last Name   First Name   Score
=======================================
Blue    Hope        Sue          17
        Woods       Bill         12
                    Total        29

Red     Doe         John         20
        Smith       Jim          10
        Ringer      Bob          18
                    Total        48
              Grand Total        77

===============================================================

Copyright © 2002 FileMaker, Inc.
All rights reserved.

Redistribution and use in source and binary forms, with or
without modification, are permitted provided that the following
conditions are met:

* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in 
  the documentation and/or other materials provided with the
  distribution.

* Neither the name of the FileMaker, Inc. nor the names of its 
  contributors may be used to endorse or promote products derived
  from this software without specific prior written
  permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. 
IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR 
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
    
===============================================================
-->
<!--
Use the Meunchian method to group the exported data based on data 
in the first field.
-->
	<xsl:key name="groupKey" match="fmp:FMPXMLRESULT/fmp:RESULTSET/fmp:ROW" use="fmp:COL[1]/fmp:DATA"/>
<!--
Same idea, stuff the database name into a variable for use elsewhere.
-->
	<xsl:variable name="dbName">
		<xsl:value-of select="fmp:FMPXMLRESULT/fmp:DATABASE/@NAME"/>
	</xsl:variable>
<!--
Change these variables to alter the appearance of the table.
-->
	<xsl:variable name="rowColor">
		<xsl:value-of select="'#cccccc'"/>
	</xsl:variable>
	<xsl:variable name="groupColor">
		<xsl:value-of select="'#999999'"/>
	</xsl:variable>
	<xsl:variable name="headerColor">
		<xsl:value-of select="'#9999cc'"/>
	</xsl:variable>
	<xsl:variable name="footerColor">
		<xsl:value-of select="'#9999cc'"/>
	</xsl:variable>
<!--
Count all the children in the METADATA block, this gives us the
number of columns in the database.
-->
	<xsl:variable name="dbNumCols">
		<xsl:value-of select="count(fmp:FMPXMLRESULT/fmp:METADATA/child::*) "/>
	</xsl:variable>
	
<!--
The data type of the last column in the export determines whether 
or not the data is counted or totaled.
-->
	<xsl:variable name="type">
		<xsl:value-of select="fmp:FMPXMLRESULT/fmp:METADATA/fmp:FIELD[last()]/@TYPE"/>
	</xsl:variable>
	<!--
	-->
	<xsl:template match="fmp:FMPXMLRESULT">
		<html>
			<head>
				<title>
					<xsl:value-of select="$dbName"/>
				</title>
			</head>
			<body>
				<table cellSpacing="1" cellPadding="3" width="100%" border="0">
					<xsl:call-template name="tableHeader"/>
<!--
For every unique value in column one, create a new group
-->
					<xsl:for-each select="fmp:RESULTSET/fmp:ROW[generate-id(.)=generate-id(key('groupKey', fmp:COL[1]/fmp:DATA)[1])]">
<!--
Sort the groups
-->
						<xsl:sort select="fmp:COL[1]/fmp:DATA" data-type="text"/>
						<xsl:variable name="currentGroup" select="key('groupKey', fmp:COL[1]/fmp:DATA)"/>
						<xsl:for-each select="$currentGroup">
<!--
Sort the members of each group by column two
-->
							<xsl:sort select="fmp:COL[2]/fmp:DATA" data-type="text"/>
							<xsl:if test="position()=1">
								<xsl:call-template name="groupRow"/>
							</xsl:if>
							<xsl:call-template name="makeRow"/>
							<xsl:if test="position()=count($currentGroup)">

									<!--
									subSumRow
								
									Either counts or totals the members of each group.
									-->
								
										<tr>
										<td align="right" valign="middle">
											<xsl:attribute name="bgcolor"><xsl:value-of select="$rowColor"/></xsl:attribute>
											<xsl:attribute name="colspan"><xsl:value-of select="$dbNumCols - 1"/></xsl:attribute>
											<xsl:choose>
												<xsl:when test="$type = 'NUMBER'">
													<xsl:text>Total:</xsl:text>
												</xsl:when>
												<xsl:otherwise>
													<xsl:text>Count:</xsl:text>
												</xsl:otherwise>
											</xsl:choose>
										</td>
										<td align="middle" valign="middle">
											<xsl:attribute name="bgcolor"><xsl:value-of select="$rowColor"/></xsl:attribute>
											<xsl:variable name="result">
												<xsl:choose>
													<xsl:when test="$type = 'NUMBER'">
														<xsl:value-of select="sum($currentGroup/fmp:COL[last()]/fmp:DATA)"/>  
													</xsl:when>
													<xsl:otherwise>
														<xsl:value-of select="count($currentGroup)"/>
													</xsl:otherwise>
												</xsl:choose>
											</xsl:variable>
											<font size="4">
												<xsl:value-of select="$result"/>
											</font>
										</td>
									</tr>
									<tr>
										<td>
											<xsl:attribute name="colspan"><xsl:value-of select="$dbNumCols"/></xsl:attribute>
											<xsl:attribute name="bgcolor"><xsl:value-of select="$rowColor"/></xsl:attribute>
											<font size="4">
												<xsl:text disable-output-escaping="yes">&amp;nbsp;</xsl:text>
											</font>
										</td>
									</tr>
							

							</xsl:if>
						</xsl:for-each>
					</xsl:for-each>
					<tr>
						<xsl:choose>
							<xsl:when test="$type = 'NUMBER'">
<!--
A field of type NUMBER results in totals
-->
								<td align="right" valign="middle">
									<xsl:attribute name="bgcolor"><xsl:value-of select="$footerColor"/></xsl:attribute>
									<xsl:attribute name="colspan"><xsl:value-of select="$dbNumCols - 1"/></xsl:attribute>
									<font size="4">
										<xsl:text>Grand Total:</xsl:text>
									</font>
								</td>
								<td align="middle" valign="middle">
									<xsl:attribute name="bgcolor"><xsl:value-of select="$footerColor"/></xsl:attribute>
									<font size="4">
										<xsl:value-of select="sum(fmp:RESULTSET/fmp:ROW/fmp:COL[last()]/fmp:DATA)"/>
									</font>
								</td>
							</xsl:when>
							<xsl:otherwise>
<!--
Otherwise, simply count the number of items in each group
and total them
-->
								<td align="right" valign="middle">
									<xsl:attribute name="bgcolor"><xsl:value-of select="$footerColor"/></xsl:attribute>
									<xsl:attribute name="colspan"><xsl:value-of select="$dbNumCols - 1"/></xsl:attribute>
									<font size="4"/>
									<xsl:text>Total Count:</xsl:text>
								</td>
								<td align="middle" valign="middle">
									<xsl:attribute name="bgcolor"><xsl:value-of select="$footerColor"/></xsl:attribute>
									<font size="4">
										<xsl:value-of select="count(fmp:RESULTSET/fmp:ROW/fmp:COL[1]/fmp:DATA)"/>
									</font>
								</td>
							</xsl:otherwise>
						</xsl:choose>
					</tr>
				</table>
			</body>
		</html>
	</xsl:template>
<!--
Template: tableHeader
	
The database name spans the entire table.
Individual field names head each column.
-->
	<xsl:template name="tableHeader">
		<tr>
			<th height="50" align="left" valign="middle">
				<xsl:attribute name="bgcolor"><xsl:value-of select="$headerColor"/></xsl:attribute>
				<xsl:attribute name="colspan"><xsl:value-of select="$dbNumCols"/></xsl:attribute>
				<font size="4">
					<DIV align="center">
						<xsl:value-of select="fmp:DATABASE/@NAME"/>
					</DIV>
				</font>
			</th>
		</tr>
		<tr>
			<xsl:for-each select="/*/*/fmp:FIELD">
				<th align="middle" valign="middle">
					<xsl:attribute name="bgcolor"><xsl:value-of select="$headerColor"/></xsl:attribute>
					<font size="4">
						<xsl:value-of select="@NAME"/>
					</font>
				</th>
			</xsl:for-each>
		</tr>
	</xsl:template>
<!--
Template: makeRow
	
Builds the individual table rows.
Center the data in the last cell if the field is a number.
Add spaces to the output to move the data away from table
cell borders.
-->
	<xsl:template name="makeRow">
		<tr>
			<td valign="middle">
				<xsl:attribute name="bgcolor"><xsl:value-of select="$rowColor"/></xsl:attribute>
				<font size="4">
					<xsl:text disable-output-escaping="yes">&amp;nbsp;</xsl:text>
				</font>
			</td>
			<xsl:variable name="alignData">
				<xsl:choose>
					<xsl:when test="$type = 'NUMBER'">
						<xsl:value-of select="'middle'"/>
					</xsl:when>
					<xsl:otherwise>
						<xsl:value-of select="'left'"/>
					</xsl:otherwise>
				</xsl:choose>
			</xsl:variable>
			<xsl:for-each select="fmp:COL">
				<xsl:choose>
					<xsl:when test="position() = 1"/>
					<xsl:when test="position() = last()">
						<td valign="middle">
							<xsl:attribute name="align"><xsl:value-of select="$alignData"/></xsl:attribute>
							<xsl:attribute name="bgcolor"><xsl:value-of select="$rowColor"/></xsl:attribute>
							<xsl:choose>
								<xsl:when test="$type = 'NUMBER'">
									<xsl:text/>
								</xsl:when>
								<xsl:otherwise>
									<xsl:text disable-output-escaping="yes">&amp;nbsp;</xsl:text>
								</xsl:otherwise>
							</xsl:choose>
							<xsl:value-of select="fmp:DATA"/>
						</td>
					</xsl:when>
					<xsl:otherwise>
						<td align="left" valign="middle">
							<xsl:attribute name="bgcolor"><xsl:value-of select="$rowColor"/></xsl:attribute>
							<xsl:text disable-output-escaping="yes">&amp;nbsp;</xsl:text>
							<xsl:value-of select="fmp:DATA"/>
						</td>
					</xsl:otherwise>
				</xsl:choose>
			</xsl:for-each>
		</tr>
	</xsl:template>
<!--
Template: groupRow
	
Start each group with a row that spans the entire table 
and is shaded differently.
-->
	<xsl:template name="groupRow">
		<tr>
			<td align="left">
				<xsl:attribute name="bgcolor"><xsl:value-of select="$groupColor"/></xsl:attribute>
				<xsl:attribute name="colspan"><xsl:value-of select="$dbNumCols"/></xsl:attribute>
				<font size="4">
					<xsl:text disable-output-escaping="yes">&amp;nbsp;&amp;nbsp;</xsl:text>
					<xsl:value-of select="fmp:COL[1]/fmp:DATA"/>
				</font>
			</td>
		</tr>
	</xsl:template>

</xsl:stylesheet>
